<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once '../includes/functions.php';

// Check if user is logged in and has permission
if (!isLoggedIn() || !hasPermission('manage_products')) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit;
}

header('Content-Type: application/json');

try {
    // Get form data
    $product_id = intval($_POST['product_id'] ?? 0);
    $quantity = intval($_POST['quantity'] ?? 0);
    $reason = trim($_POST['reason'] ?? '');

    if (!$product_id || !$quantity || empty($reason)) {
        throw new Exception('All fields are required');
    }

    // Start transaction
    $pdo->beginTransaction();

    // Get current stock
    $stmt = $pdo->prepare("SELECT quantity, name FROM products WHERE id = ?");
    $stmt->execute([$product_id]);
    $product = $stmt->fetch();

    if (!$product) {
        throw new Exception('Product not found');
    }

    // Calculate new quantity
    $new_quantity = $product['quantity'] + $quantity;

    // Check if removing stock would result in negative quantity
    if ($new_quantity < 0) {
        throw new Exception('Cannot remove more stock than available');
    }

    // Update product quantity
    $stmt = $pdo->prepare("UPDATE products SET quantity = ? WHERE id = ?");
    $stmt->execute([$new_quantity, $product_id]);

    // Log the adjustment
    $stmt = $pdo->prepare("
        INSERT INTO stock_adjustments 
        (product_id, quantity, reason, adjusted_by, adjusted_at) 
        VALUES (?, ?, ?, ?, NOW())
    ");
    $stmt->execute([$product_id, $quantity, $reason, $_SESSION['user_id']]);

    // Log to system logs
    $details = sprintf(
        "Product: %s (ID: %d), Adjustment: %+d, New Stock: %d, Reason: %s, By: %s (User ID: %d)",
        $product['name'],
        $product_id,
        $quantity,
        $new_quantity,
        $reason,
        $_SESSION['username'],
        $_SESSION['user_id']
    );
    logSystemAction('stock_adjusted', $details);

    // Commit transaction
    $pdo->commit();
    echo json_encode(['success' => true]);

} catch (Exception $e) {
    // Rollback transaction on error
    $pdo->rollBack();
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
} 